<?php
/**
 * Server-side file used to retrieve arbitrary language tokens through AJAX.
 *
 * @see Application.Modules.Language (JavaScript)
 *
 * @package interspire.iem
 */

// Make sure that the IEM controller does NOT redirect request.
define('IEM_NO_CONTROLLER', true);

// Include the index file
require_once dirname(__FILE__) . '/index.php';

/**
 * TokenFactory
 * This class manages the retrieval of language tokens from nominated files.
 *
 * @package SendStudio
 */
class TokenFactory
{
	/**
	 * The path to look for language token files.
	 * 
	 * @var String $base_path
	 */
	private $base_path;
	
	/**
	 * The default language path for fallback.
	 * 
	 * @var String $default_path
	 */
	private $default_path;

	/**
	 * __construct
	 * Sets the base path where to look for language token files.
	 *
	 * @return Void Does not return anything.
	 */
	public function __construct()
	{
		$lang_folder = IEM_PATH . '/language';
		$user_lang_folder = 'default';

		// ----- Get user language preference
			$user = IEM::getCurrentUser();
			$temp = $user->user_language;

			if (!empty($temp) && is_dir("{$lang_folder}/{$temp}")) {
				$user_lang_folder = $temp;
			}

			unset($temp);
			unset($user);
		// -----

		$this->base_path = "{$lang_folder}/{$user_lang_folder}";
		$this->default_path = "{$lang_folder}/default";
	}

	/**
	 * LookIn
	 * Calls require to load the langauge files to look for language tokens in.
	 * 
	 * @param Array $files
	 * 
	 * @return Bool
	 */
	public function LookIn($files)
	{
		if (!is_array($files)) {
			return false;
		}
		$files = array_unique($files);
		foreach ($files as $filename) {
			if ($this->_Valid($filename)) {
				$file_path = $this->base_path . '/' . $filename;
				if (file_exists($file_path)) {
					require_once($file_path);
				} else {
					// Try to load from default language if file doesn't exist in user's language
					$default_file_path = $this->default_path . '/' . $filename;
					if (file_exists($default_file_path)) {
						require_once($default_file_path);
					}
				}
			}
		}
		return true;
	}

	/**
	 * GetTokens
	 * Returns the language token values. Invalid tokens will have the value of
	 * the empty string.
	 *
	 * @param Array $tokens The list of language tokens without the preceing 'LNG_'
	 *
	 * @return Array|False The list of language token values in the form array($token => $value);
	 */
	public function GetTokens($tokens)
	{
		if (!is_array($tokens)) {
			return false;
		}
		$tokens = array_unique($tokens);
		$result = [];
		foreach ($tokens as $token) {
			// Use GetLang which now has fallback capability
			$result[$token] = GetLang($token, '');
		}
		return $result;
	}

	/**
	 * _Valid
	 * Checks if the filename is valid and safe to include.
	 * 
	 * @param String $filename
	 * 
	 * @return Bool
	 */
	private function _Valid($filename)
	{
		// Check for path traversal attempts
		if (strpos($filename, '..') !== false) {
			return false;
		}
		
		// Check if the filename has a valid format
		if (!preg_match('/^[a-z0-9_\-\.]+\.php$/i', $filename)) {
			return false;
		}
		
		// Check if the file exists in either the user's language path or the default path
		return (file_exists($this->base_path . '/' . $filename) || file_exists($this->default_path . '/' . $filename));
	}
}

header('Content-type: application/json');

$factory = new TokenFactory();
if ($factory->LookIn($_POST['files'])) {
	$tokens = $factory->GetTokens($_POST['tokens']);
	$response = [
		'status' => 'OK',
		'tokens' => GetJSON($tokens)
	];
	echo GetJSON($response);
} else {
	echo GetJSON(['status' => 'Failed']);
}
